"use client";

import Link from "next/link";
import { useState } from "react";
import { useSearchParams } from "next/navigation";
import type { Company } from "@prisma/client";
import AddCompanyModal from "./AddCompanyModal";

export default function Sidebar({ companies }: { companies: Company[] }) {
    const searchParams = useSearchParams();
    const currentCompanyId = searchParams.get("companyId");
    const [showModal, setShowModal] = useState(false);
    const [isMobileOpen, setIsMobileOpen] = useState(false);

    const toggleMobileMenu = () => setIsMobileOpen(!isMobileOpen);

    return (
        <>
            {/* Mobile Hamburger Button */}
            <button
                onClick={toggleMobileMenu}
                className="lg:hidden fixed top-4 left-4 z-50 p-2 bg-white rounded-md shadow-md border border-zinc-200 text-zinc-600"
            >
                {isMobileOpen ? (
                    <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12" /></svg>
                ) : (
                    <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M4 6h16M4 12h16M4 18h16" /></svg>
                )}
            </button>

            {/* Backdrop for Mobile */}
            {isMobileOpen && (
                <div
                    className="fixed inset-0 bg-black/50 z-40 lg:hidden"
                    onClick={() => setIsMobileOpen(false)}
                />
            )}

            {/* Sidebar Container */}
            <div className={`
                fixed inset-y-0 left-0 z-40 w-64 bg-zinc-50 border-r border-zinc-200 transform transition-transform duration-200 ease-in-out lg:translate-x-0 lg:static lg:h-screen lg:flex lg:flex-col
                ${isMobileOpen ? 'translate-x-0' : '-translate-x-full'}
            `}>
                <div className="flex flex-col h-full p-4">
                    <div className="flex items-center justify-between mb-6">
                        <h2 className="font-bold text-zinc-900 pl-8 lg:pl-0">Firmalar</h2>
                        <div className="flex gap-2">
                            <Link href="/calendar" className="p-1 rounded-md hover:bg-zinc-200 text-zinc-600" title="Takvim">
                                📅
                            </Link>
                            <button
                                onClick={() => setShowModal(true)}
                                className="p-1 rounded-md hover:bg-zinc-200 text-zinc-600 font-bold"
                                title="Yeni Firma Ekle"
                            >
                                +
                            </button>
                        </div>
                    </div>

                    <nav className="space-y-1 flex-1 overflow-y-auto">
                        {companies.length === 0 && (
                            <p className="text-sm text-zinc-500 italic p-2">Henüz firma yok.</p>
                        )}
                        {companies.map((company) => {
                            const isActive = currentCompanyId === company.id;
                            return (
                                <Link
                                    key={company.id}
                                    href={`/?companyId=${company.id}`}
                                    onClick={() => setIsMobileOpen(false)}
                                    className={`block px-3 py-2 rounded-md text-sm font-medium transition-colors ${isActive
                                        ? "bg-blue-100 text-blue-900"
                                        : "text-zinc-700 hover:bg-zinc-200"
                                        }`}
                                >
                                    {company.name}
                                </Link>
                            )
                        })}
                    </nav>

                    <div className="pt-4 mt-4 border-t border-zinc-200 text-xs text-zinc-400 text-center">
                        Artiz Creative v1.0
                    </div>
                </div>
            </div>

            {showModal && <AddCompanyModal onClose={() => setShowModal(false)} />}
        </>
    );
}
