"use client";

import { useState } from "react";
import Link from "next/link";
import Sidebar from "@/components/Sidebar";
import Toast, { ToastType } from "@/components/Toast";
import type { Post, Company } from "@prisma/client";

// Helper to get days in month
function getDaysInMonth(year: number, month: number) {
    return new Date(year, month + 1, 0).getDate();
}

// Helper to get day of week for first day (0=Sunday, but we want 0=Monday)
function getFirstDayOfMonth(year: number, month: number) {
    const day = new Date(year, month, 1).getDay();
    return day === 0 ? 6 : day - 1;
}

export default function CalendarPage({ posts, companies }: { posts: (Post & { company: Company })[], companies: Company[] }) {
    const [currentDate, setCurrentDate] = useState(new Date());
    const [selectedDay, setSelectedDay] = useState<string | null>(null); // YYYY-MM-DD
    const [toast, setToast] = useState<{ message: string; type: ToastType; isVisible: boolean }>({
        message: "",
        type: "success",
        isVisible: false,
    });

    const showToast = (message: string, type: ToastType = "success") => {
        setToast({ message, type, isVisible: true });
    };

    const handleCopy = async (text: string) => {
        try {
            await navigator.clipboard.writeText(text);
            showToast("Kopyalandı!");
        } catch (err) {
            showToast("Kopyalama başarısız", "error");
        }
    };

    const year = currentDate.getFullYear();
    const month = currentDate.getMonth();

    const daysInMonth = getDaysInMonth(year, month);
    const startOffset = getFirstDayOfMonth(year, month);

    const days = Array.from({ length: daysInMonth }, (_, i) => i + 1);
    const blanks = Array.from({ length: startOffset }, (_, i) => i);

    const monthNames = ["Ocak", "Şubat", "Mart", "Nisan", "Mayıs", "Haziran", "Temmuz", "Ağustos", "Eylül", "Ekim", "Kasım", "Aralık"];

    // Group posts
    const postsByDate: Record<string, typeof posts> = {};
    posts.forEach(post => {
        if (!post.scheduledDate) return;
        const dateKey = new Date(post.scheduledDate).toISOString().split('T')[0];
        if (!postsByDate[dateKey]) postsByDate[dateKey] = [];
        postsByDate[dateKey].push(post);
    });

    const changeMonth = (offset: number) => {
        setCurrentDate(new Date(year, month + offset, 1));
    };

    const selectedDayPosts = selectedDay ? postsByDate[selectedDay] || [] : [];

    return (
        <div className="flex bg-white min-h-screen">
            <div className="hidden lg:block">
                <Sidebar companies={companies} />
            </div>

            <Toast
                message={toast.message}
                type={toast.type}
                isVisible={toast.isVisible}
                onClose={() => setToast(prev => ({ ...prev, isVisible: false }))}
            />

            {/* Daily View Modal */}
            {selectedDay && (
                <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50 backdrop-blur-sm p-4" onClick={() => setSelectedDay(null)}>
                    <div className="w-full max-w-2xl bg-white rounded-xl shadow-2xl p-6 relative max-h-[80vh] overflow-y-auto" onClick={e => e.stopPropagation()}>
                        <button
                            onClick={() => setSelectedDay(null)}
                            className="absolute top-4 right-4 text-zinc-400 hover:text-zinc-600"
                        >
                            ✕
                        </button>
                        <h2 className="text-xl font-bold mb-6 text-zinc-900 border-b pb-2 border-zinc-100">
                            {new Date(selectedDay).toLocaleDateString("tr-TR", { day: 'numeric', month: 'long', year: 'numeric' })}
                        </h2>

                        {selectedDayPosts.length === 0 ? (
                            <p className="text-zinc-500 text-center py-8">Bu tarih için planlanmış içerik yok.</p>
                        ) : (
                            <div className="space-y-4">
                                {selectedDayPosts.map(post => (
                                    <div key={post.id} className="border border-zinc-200 rounded-lg p-4 bg-zinc-50">
                                        <div className="flex items-center justify-between mb-2">
                                            <div className="flex items-center gap-2">
                                                <span className="font-bold text-zinc-900 text-sm">{post.company.name}</span>
                                                <span className={`text-[10px] px-1.5 py-0.5 rounded border ${post.type === 'REELS' ? 'bg-pink-100 text-pink-700 border-pink-200' : 'bg-blue-100 text-blue-700 border-blue-200'}`}>
                                                    {post.type}
                                                </span>
                                            </div>
                                            <button
                                                onClick={() => handleCopy(post.fullText)}
                                                className="text-xs font-semibold text-white bg-blue-600 hover:bg-blue-700 px-3 py-1.5 rounded transition-colors"
                                            >
                                                Kopyala
                                            </button>
                                        </div>
                                        <div className="text-sm text-zinc-600 whitespace-pre-wrap mb-2">
                                            {post.content}
                                        </div>
                                        {post.subject && (
                                            <div className="text-xs text-zinc-500 italic">
                                                Konu: {post.subject}
                                            </div>
                                        )}
                                    </div>
                                ))}
                            </div>
                        )}
                    </div>
                </div>
            )}

            <div className="flex-1 p-4 lg:p-8 h-screen overflow-y-auto w-full">
                <header className="mb-8 flex flex-col sm:flex-row justify-between items-center gap-4">
                    <div className="flex items-center gap-4 order-2 sm:order-1">
                        <button onClick={() => changeMonth(-1)} className="p-2 hover:bg-zinc-100 rounded-full">←</button>
                        <h1 className="text-xl sm:text-2xl font-bold text-zinc-900 w-40 sm:w-48 text-center">
                            {monthNames[month]} {year}
                        </h1>
                        <button onClick={() => changeMonth(1)} className="p-2 hover:bg-zinc-100 rounded-full">→</button>
                    </div>
                    <Link href="/" className="order-1 sm:order-2 text-sm font-medium text-blue-600 hover:bg-blue-50 px-3 py-2 rounded-lg transition-colors bg-blue-50/50 sm:bg-transparent w-full sm:w-auto text-center">
                        &larr; Panele Dön
                    </Link>
                </header>

                <div className="overflow-x-auto pb-4">
                    <div className="min-w-[800px] grid grid-cols-7 gap-px bg-zinc-200 border border-zinc-200 rounded-lg overflow-hidden shadow-sm">
                        {["Pzt", "Sal", "Çar", "Per", "Cum", "Cmt", "Paz"].map(day => (
                            <div key={day} className="bg-zinc-50 p-3 text-center text-sm font-semibold text-zinc-500">
                                {day}
                            </div>
                        ))}

                        {blanks.map(blank => (
                            <div key={`blank-${blank}`} className="bg-white min-h-[120px]" />
                        ))}

                        {days.map(day => {
                            const dateKey = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
                            const dayPosts = postsByDate[dateKey] || [];
                            const isToday = dateKey === new Date().toISOString().split('T')[0];

                            return (
                                <div
                                    key={day}
                                    onClick={() => setSelectedDay(dateKey)}
                                    className={`bg-white min-h-[120px] p-2 border-t border-zinc-100 cursor-pointer transition-colors hover:bg-blue-50/50 group relative`}
                                >
                                    <div className="flex justify-between items-start">
                                        <span className={`text-sm font-medium h-7 w-7 flex items-center justify-center rounded-full ${isToday ? 'bg-blue-600 text-white shadow-md' : 'text-zinc-700'}`}>
                                            {day}
                                        </span>
                                        {dayPosts.length > 0 && (
                                            <span className="text-[10px] bg-zinc-100 text-zinc-500 px-1.5 rounded-full font-medium">
                                                {dayPosts.length}
                                            </span>
                                        )}
                                    </div>

                                    <div className="mt-2 space-y-1">
                                        {dayPosts.slice(0, 3).map(post => (
                                            <div key={post.id} className="text-[10px] p-1.5 rounded border border-zinc-100 bg-zinc-50 truncate shadow-sm flex items-center gap-1">
                                                <div className={`w-1.5 h-1.5 rounded-full flex-shrink-0 ${post.type === 'REELS' ? 'bg-pink-500' : 'bg-blue-500'}`} />
                                                <span className="font-semibold text-zinc-900">{post.company.name}</span>
                                            </div>
                                        ))}
                                        {dayPosts.length > 3 && (
                                            <div className="text-[10px] text-center text-zinc-400 font-medium">
                                                + {dayPosts.length - 3} daha
                                            </div>
                                        )}
                                    </div>
                                </div>
                            );
                        })}
                    </div>
                </div>
            </div>
        </div>
    );
}
